//
// Copyright 2011 Justin.tv. All rights reserved.
//

#import "TWTableViewCell.h"
#import "UIFont+Twitch.h"

NSString *const TWTableViewCellReuseIdentifier = @"TWTableViewCellReuseIdentifier";

@implementation TWTableViewCell {
	
	UIView *_bottomLine;
	UIView *_customAccessoryView;
	UITableViewCellAccessoryType _customAccessoryType;
}

#define TWTableViewCellDefaultMargin 10.f

+ (NSString *)reuseIdentifier
{
	return TWTableViewCellReuseIdentifier;
}

+ (CGFloat)preferredHeight
{
	return 43.f;
}

- (void)dealloc {
    
    [[NSNotificationCenter defaultCenter] removeObserver:self];
}

- (void)commonInitialization
{
	_textLabelPositionAdjustment = _detailTextLabelPositionAdjustment = CGPointMake(0.f, 0.f);
	_bottomSeparatorColor = [UIColor twitchCellSeparatorColor];
	_accessoryTintColor	= [UIColor twitchCellAccessoryColor];
	_imagePadding = TWTableViewCellDefaultMargin;
	_accessoryPadding = UIEdgeInsetsMake(0.f, TWTableViewCellDefaultMargin, 0.f, TWTableViewCellDefaultMargin);
	_minimumTextIndentation = TWTableViewCellDefaultMargin;
	_imageSize = CGSizeZero;
	_shouldDownloadRedirectedImages = YES;
	
	// Set default autoresizing parameters
	self.autoresizingMask = UIViewAutoresizingFlexibleWidth;
	self.autoresizesSubviews = YES;
	self.textLabel.lineBreakMode = NSLineBreakByTruncatingTail;
	self.textLabel.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleBottomMargin | UIViewAutoresizingFlexibleTopMargin;
	self.detailTextLabel.lineBreakMode = NSLineBreakByTruncatingTail;
	self.detailTextLabel.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleBottomMargin | UIViewAutoresizingFlexibleTopMargin;
	
	// Set default image view parameters
	self.imageView.autoresizingMask = UIViewAutoresizingFlexibleRightMargin;
	self.imageView.contentMode = UIViewContentModeCenter;
	self.imageView.clipsToBounds = YES;
	
	// Set default fonts and colors
	self.textLabel.font = [UIFont twitchCellTextFont];
	self.detailTextLabel.font = [UIFont twitchCellDetailTextFont];
	self.textLabel.textColor = [UIColor twitchCellTextColor];
	self.detailTextLabel.textColor = [UIColor twitchCellDetailTextColor];
	self.textLabel.backgroundColor = [UIColor clearColor];
	self.detailTextLabel.backgroundColor = [UIColor clearColor];
	
	[self prepareForReuse];
}

- (id)initWithStyle:(UITableViewCellStyle)style reuseIdentifier:(NSString *)reuseIdentifier
{
    self = [super initWithStyle:UITableViewCellStyleSubtitle reuseIdentifier:reuseIdentifier];
    if (self) {
		[self commonInitialization];
    }
    return self;
}

- (id)init
{
	if ((self = [super initWithStyle:UITableViewCellStyleSubtitle reuseIdentifier:[[self class] reuseIdentifier]])) {
		[self commonInitialization];
	}
	
	return self;
}

#pragma mark -

- (void) setShowsBottomSeparator:(BOOL) shouldShowBottomSeparator {
	if (shouldShowBottomSeparator == _showsBottomSeparator)
		return;
	
	_showsBottomSeparator = shouldShowBottomSeparator;
	
	if (_showsBottomSeparator) {
		_bottomLine = [[UIView alloc] initWithFrame:CGRectZero];
		_bottomLine.backgroundColor = _bottomSeparatorColor;
		_bottomLine.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleTopMargin;
		[self.contentView addSubview:_bottomLine];
		[self setNeedsLayout];
		
	} else {
		[_bottomLine removeFromSuperview];
		_bottomLine = nil;
	}
}

#pragma mark -

- (void) prepareForReuse {
	[super prepareForReuse];
	
	self.accessibilityLabel = nil;
	
	if (self.imageURL) {
		self.imageURL = nil;
		[[NSNotificationCenter defaultCenter] removeObserver:self];
	}
	if (self.image) {
		self.imageView.image = nil;
	}
}

- (void) layoutSubviews {
	
	CGFloat textLeftMargin = 0.f;
	CGFloat textRightMargin = self.frame.size.width - TWTableViewCellDefaultMargin;
	BOOL hasImage = CGSizeEqualToSize(_imageSize, CGSizeZero) ? NO : YES;
	BOOL hasDetailText = self.detailTextLabel.text.length > 0 ? YES : NO;
	
	// Resize the content view and selected background view
	CGRect cellRect = CGRectMake(0.f, 0.f, self.frame.size.width, self.frame.size.height);
	self.contentView.frame = cellRect;
	self.selectedBackgroundView.frame = cellRect;
	
	// Position the image
	if (hasImage) {
		
		CGFloat imageVerticalOrigin = _contentVerticalAlignment == TWTableViewCellContentVerticalAlignmentTop ? 0.f : (_contentVerticalAlignment == TWTableViewCellContentVerticalAlignmentBottom ? self.frame.size.height - _imageSize.height : floorf((self.frame.size.height - _imageSize.height) / 2.f));
		CGRect imageFrame = CGRectMake(_imagePadding, imageVerticalOrigin, _imageSize.width, _imageSize.height);
		self.imageView.frame = imageFrame;
		
		textLeftMargin = _imagePadding * 2.f + _imageSize.width;
	}
	
	// Position our custom accessory view
	if (_customAccessoryView) {
		
		_customAccessoryView.frame = CGRectMake(self.frame.size.width - _accessoryPadding.right - _customAccessoryView.frame.size.width, floorf((self.frame.size.height - self.accessoryView.frame.size.height) / 2.f), _customAccessoryView.frame.size.width, _customAccessoryView.frame.size.height);
		textRightMargin = _customAccessoryView.frame.origin.x - _accessoryPadding.left;
	}
	
	if (textLeftMargin < _minimumTextIndentation) textLeftMargin = _minimumTextIndentation;
	if (textRightMargin > (self.frame.size.width - _minimumTextIndentation)) textRightMargin = self.frame.size.width - _minimumTextIndentation;
	
	// Position the text labels
	[self.textLabel sizeToFit];
	CGFloat textLabelVerticalOrigin = floorf((self.frame.size.height - self.textLabel.frame.size.height) / 2.f);
	if (hasDetailText) {
		[self.detailTextLabel sizeToFit];
		self.detailTextLabel.frame = CGRectMake(textLeftMargin + _detailTextLabelPositionAdjustment.x, floorf(self.frame.size.height / 2.f) + _detailTextLabelPositionAdjustment.y, textRightMargin - textLeftMargin, self.detailTextLabel.frame.size.height);
		textLabelVerticalOrigin = floorf(self.frame.size.height / 2.f) - self.textLabel.frame.size.height;
	}
	self.textLabel.frame = CGRectMake(textLeftMargin + _textLabelPositionAdjustment.x, textLabelVerticalOrigin + _textLabelPositionAdjustment.y, textRightMargin - textLeftMargin, self.textLabel.frame.size.height);
	
	// Resize our custom separator
	if (_bottomLine) {
		
		CGFloat bottomSeparatorHeight = 1.f / [UIScreen mainScreen].scale;
		_bottomLine.frame = CGRectMake(0.f, self.frame.size.height - bottomSeparatorHeight, self.frame.size.width, bottomSeparatorHeight);
	}
}

- (void)cacheDidNotDownloadResource:(NSNotification *)notification
{
    if (self.imageURL && [[[notification userInfo] objectForKey:@"key"] isEqualToString:self.imageURL]) {
		
        [[NSNotificationCenter defaultCenter] removeObserver:self];
        self.imageURL = nil;
    }
}

- (void)cacheDidDownloadResource:(NSNotification *)notification
{
    if (self.imageURL && [[[notification userInfo] objectForKey:@"key"] isEqualToString:self.imageURL]) {
		
		[self setImage:[[notification userInfo] objectForKey:@"object"] animated:YES];
		
        [[NSNotificationCenter defaultCenter] removeObserver:self];
        self.imageURL = nil;
    }
}

- (void)loadImageAtURL:(NSString *)url fromCache:(TWCache *)cache persistently:(BOOL)persistently {
    
	UIImage *image = nil;
	
	if (self.imageURL) [[NSNotificationCenter defaultCenter] removeObserver:self];
	self.imageURL = nil;
	
	if (url) image = [cache imageAtURL:url persistently:persistently downloadIfRedirected:_shouldDownloadRedirectedImages];
	
    if (!image) {
        
		if (url) {
			
			[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(cacheDidDownloadResource:) name:TWCacheDidDownloadObject object:cache];
			[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(cacheDidNotDownloadResource:) name:TWCacheDidNotDownloadObject object:cache];
		}
		self.imageURL = url;
		
		if (_placeholderImageName) image = [UIImage imageFromFrameworkBundleNamed:_placeholderImageName];
    }
    
	[self setImage:image animated:NO];
}

- (void)setAccessoryType:(UITableViewCellAccessoryType)accessoryType
{
	_customAccessoryType = accessoryType;
	
	UIImage *accessoryImage = nil;
	
	if (accessoryType == UITableViewCellAccessoryDisclosureIndicator) {
		
		// Set the accessory view to the custom disclosure indicator
		accessoryImage = [UIImage imageFromFrameworkBundleNamed:@"glyph_chevron_normal"];
	}
	else if (accessoryType == UITableViewCellAccessoryCheckmark) {
		
		// Set the accessory view to the custom disclosure indicator
		accessoryImage = [UIImage imageFromFrameworkBundleNamed:@"glyph_checkmark"];
	}
	
	if (accessoryImage) {
		
		self.accessoryView = [[UIImageView alloc] initWithImage:[accessoryImage imageWithRenderingMode:UIImageRenderingModeAlwaysTemplate]];
	}
	else self.accessoryView = nil;
}

- (UITableViewCellAccessoryType)accessoryType
{
	return _customAccessoryType;
}

- (void)setAccessoryView:(UIView *)accessoryView
{
	if (_customAccessoryView != accessoryView) {
		
		if (_customAccessoryView) [_customAccessoryView removeFromSuperview];
		_customAccessoryView = accessoryView;
		
		if (accessoryView) {
			
			if (_accessoryTintColor) {
				
				accessoryView.tintAdjustmentMode = UIViewTintAdjustmentModeNormal;
				accessoryView.tintColor = _accessoryTintColor;
			}
			[self.contentView addSubview:accessoryView];
		}
		[self setNeedsLayout];
	}
}

- (UIView *)accessoryView
{
	return _customAccessoryView;
}

- (void)setImage:(UIImage *)image animated:(BOOL)animated
{
	[UIView transitionWithView:self.imageView
					  duration:animated ? .25 : 0.
					   options:(UIViewAnimationOptions)(UIViewAnimationOptionTransitionCrossDissolve | UIViewAnimationOptionBeginFromCurrentState)
					animations:^{
						self.imageView.image = image;
					} completion:nil];
}

- (void)setImage:(UIImage *)image
{
	if (self.imageURL) {
		self.imageURL = nil;
		[[NSNotificationCenter defaultCenter] removeObserver:self];
	}
	self.imageView.image = image;
}

- (UIImage *)image
{
	return self.imageView.image;
}

- (void)setImageSize:(CGSize)imageSize
{
	if (!CGSizeEqualToSize(_imageSize, imageSize)) {
		
		_imageSize = imageSize;
		[self setNeedsLayout];
	}
}

- (void)setAccessoryPadding:(UIEdgeInsets)accessoryPadding
{
	if (!UIEdgeInsetsEqualToEdgeInsets(_accessoryPadding, accessoryPadding)) {
		
		_accessoryPadding = accessoryPadding;
		[self setNeedsLayout];
	}
}

- (void)setMinimumTextIndentation:(CGFloat)minimumTextIndentation
{
	if (_minimumTextIndentation > minimumTextIndentation || _minimumTextIndentation < minimumTextIndentation) {
		
		_minimumTextIndentation = minimumTextIndentation;
		[self setNeedsLayout];
	}
}

- (void)setAccessoryTintColor:(UIColor *)accessoryTintColor
{
	_accessoryTintColor = accessoryTintColor;
	if (_customAccessoryView)  {
		
		_customAccessoryView.tintAdjustmentMode = accessoryTintColor ? UIViewTintAdjustmentModeNormal : UIViewTintAdjustmentModeAutomatic;
		_customAccessoryView.tintColor = accessoryTintColor;
	}
}

- (void)setBottomSeparatorColor:(UIColor *)separatorColor
{
	_bottomSeparatorColor = separatorColor;
	if (_bottomLine){
		
		_bottomLine.backgroundColor = _bottomSeparatorColor;
	}
}

- (void)setContentVerticalAlignment:(TWTableViewCellContentVerticalAlignment)contentVerticalAlignment
{
	if (_contentVerticalAlignment != contentVerticalAlignment) {
		
		_contentVerticalAlignment = contentVerticalAlignment;
		[self setNeedsLayout];
	}
}

- (void)setTextLabelPositionAdjustment:(CGPoint)textLabelPositionAdjustment
{
	if (!CGPointEqualToPoint(_textLabelPositionAdjustment, textLabelPositionAdjustment)) {
		
		_textLabelPositionAdjustment = textLabelPositionAdjustment;
		[self setNeedsLayout];
	}
}

- (void)setDetailTextLabelPositionAdjustment:(CGPoint)detailTextLabelPositionAdjustment
{
	if (!CGPointEqualToPoint(_detailTextLabelPositionAdjustment, detailTextLabelPositionAdjustment)) {
		
		_detailTextLabelPositionAdjustment = detailTextLabelPositionAdjustment;
		[self setNeedsLayout];
	}
}

@end
